#!/usr/bin/env python3
import argparse, json
from pathlib import Path
from v2_dephasing.benchmark import (
    run_uniform_suite, run_gaussian_suite, run_twopoint_suite,
    rmse, is_monotone_decreasing, write_uniform_plot
)

def main():
    p = argparse.ArgumentParser(description="Present‑Act V2 — Phase‑Noise (Pure Dephasing) Benchmark")
    p.add_argument("--outdir", default="artifacts", type=str, help="Output directory for CSV/JSON/plots")
    p.add_argument("--k", default=50000, type=int, help="Trials per jitter point")
    p.add_argument("--seed", default=20251031, type=int, help="Base RNG seed (PCG64)")
    p.add_argument("--eps-window", default=0.10, type=float, help="Half-width of windows in cycles")
    p.add_argument("--centers", nargs=2, default=[0.0, 0.5], type=float, help="Two window centers in cycles")
    p.add_argument("--uniform", action="store_true", help="Run uniform jitter suite only (if alone) or include it")
    p.add_argument("--gaussian", action="store_true", help="Run gaussian jitter suite only (if alone) or include it")
    p.add_argument("--twopoint", action="store_true", help="Run two-point jitter suite only (if alone) or include it")
    p.add_argument("--no-plot", action="store_true", help="Skip the uniform overlay plot")
    args = p.parse_args()

    outdir = Path(args.outdir); outdir.mkdir(parents=True, exist_ok=True)
    centers = (float(args.centers[0]), float(args.centers[1]))

    # Decide which to run
    flags = [args.uniform, args.gaussian, args.twopoint]
    if not any(flags):
        do_uniform = do_gaussian = do_twopoint = True
    else:
        do_uniform = args.uniform
        do_gaussian = args.gaussian
        do_twopoint = args.twopoint

    summary = {"config": {
        "K_per_point": int(args.k),
        "windows": {"centers_cycles": list(centers), "halfwidth_eps": float(args.eps_window)},
        "rng": "PCG64",
        "base_seed": int(args.seed)
    }}

    if do_uniform:
        dfU = run_uniform_suite(K=args.k, eps_window=args.eps_window, centers=centers, base_seed=args.seed)
        dfU.to_csv(outdir / "dephasing_uniform.csv", index=False)
        summary["uniform"] = {
            "rmse": rmse(dfU["pred_sinc"], dfU["visibility_norm"]),
            "monotone_decreasing": is_monotone_decreasing(dfU["visibility_norm"].values),
        }
        summary["uniform"]["pass"] = (summary["uniform"]["rmse"] < 0.01) and summary["uniform"]["monotone_decreasing"]
        if not args.no_plot:
            write_uniform_plot(dfU, outdir / "uniform_fit.png")

    if do_gaussian:
        dfG = run_gaussian_suite(K=args.k, base_seed=args.seed)
        dfG.to_csv(outdir / "dephasing_gaussian.csv", index=False)
        summary["gaussian"] = {
            "rmse": rmse(dfG["pred_exp"], dfG["visibility_norm"]),
            "pass": (rmse(dfG["pred_exp"], dfG["visibility_norm"]) < 0.01)
        }

    if do_twopoint:
        dfT = run_twopoint_suite(K=args.k, base_seed=args.seed)
        dfT.to_csv(outdir / "dephasing_twopoint.csv", index=False)
        summary["twopoint"] = {
            "rmse": rmse(dfT["pred_cos"], dfT["visibility_norm"]),
            "pass": (rmse(dfT["pred_cos"], dfT["visibility_norm"]) < 0.01)
        }

    with open(outdir / "summary.json", "w") as f:
        json.dump(summary, f, indent=2)
    print(json.dumps(summary, indent=2))

if __name__ == "__main__":
    main()
